/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: RS232Layer.cpp 13733 2020-11-06 12:34:03Z RMayer $:

  Description:
    Implementation of the physical layer for UART interfaces

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2020-11-06  Sending eDELETE notification in connector Deinit().
                This is necessary to be compatible with the netX transport toolkit for
                proper connector unload if running as a DLL.
    2019-12-09  - Reworked COM detection to improve detection speed by only checking
                  which serial "COM" ports are available.
                - Reworked Trace output to be always available on "debugview"
                - Changed file haeader to actual layout
    2011-04-18  Connector initialization may not return if registering of device notification failed
    2011-04-07  Bugfix: CRS232Interface::Start() returns wrong
                        error code if interface is disabled
                Bugfix: Serial port enumeration failed due to
                        limited access rights to windows registry
                        (initial enum changed, dynamic port changes are still affected)
                Debug traces added
    2010-11-25  Added detection of disabled ports and skip enumeration
                                    for these ports
    2010-10-12  Bugfix: Send event (overlapped IO) was not closed
    2009-11-27  Review
    2009-03-03  initial version

**************************************************************************************/

/*****************************************************************************/
/*! \file RS232Layer.cpp
*   Implementation of the physical layer for UART interfaces                 */
/*****************************************************************************/

#include "StdAfx.h"
#include <crtdbg.h>
#include <atlstr.h>
#include <setupapi.h>
#include <winioctl.h>
#include <devguid.h>

#include "RS232Layer.h"
#include "RS232Config.h"
#include "netXConnectorErrors.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_RS232 netX RS232 Connector                    */
/*! \{                                                                       */
/*****************************************************************************/

extern HANDLE       g_hModule;     /*!< Global DLL module handle */
extern CRS232Layer* g_pcRS232Layer;

/* All device classes to register notification for
    Copy from HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\DeviceClasses */

#define  PREFIX_COM_INTERFACE         "\\\\.\\"     /*!< Prefix for the serial interface name */

#define  CONNECTOR_THREAD_TIMEOUT     1000          /*!< Internal timeout for thread handling */
#define  END_THREAD_WAIT_TIMEOUT      2000          /*!< Internal timeout for thread handling */

const  TCHAR* szWindowClass    = _T("EmptyWindow"); /*!< Name of window class for serial device handling */

///////////////////////////////////////////////////////////////////////////////////////////
/// Debug trace
///////////////////////////////////////////////////////////////////////////////////////////

/* setup degung output (TRACE function) to windows "debugview" */
#define TRACE odprintf
static void __cdecl odprintf(const char *format, ...)
{
  va_list         vaformat;
  char            szVarstring[MAX_PATH];
  char            szBuffer[MAX_PATH];
  SYSTEMTIME      SystemTime;

  GetLocalTime( &SystemTime);

  /* Don't print the date information to log */
  (void)sprintf_s( szBuffer, MAX_PATH,"RS232CON: %.2d:%.2d:%.2d.%.3d: ",
                    SystemTime.wHour,
                    SystemTime.wMinute,
                    SystemTime.wSecond,
                    SystemTime.wMilliseconds);

  va_start( vaformat, format);

  DWORD  dwStrlen = (DWORD)strlen(szBuffer);

  (void)vsnprintf_s( szVarstring, MAX_PATH, MAX_PATH - dwStrlen, format, vaformat);
  va_end( vaformat);

  (void)strcat_s(szBuffer, MAX_PATH, szVarstring);

  /* Use Debugprint() for the output */
  szBuffer[strlen(szBuffer)+1] = '\0';
  szBuffer[strlen(szBuffer)] = '\n';
  OutputDebugString(szBuffer);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// GUIDs to register device manager notification for
///////////////////////////////////////////////////////////////////////////////////////////
static const GUID GUID_DEVINTERFACE_LIST[] =
{
  // GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR,

  GUID_DEVINTERFACE_COMPORT,

  // GUID_DEVINTERFACE_USB_DEVICE <usbiodef.h>
  /*{ 0xA5DCBF10, 0x6530, 0x11D2, { 0x90, 0x1F, 0x00, 0xC0, 0x4F, 0xB9, 0x51, 0xED } },*/
};

///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor of CRS232Layer
///////////////////////////////////////////////////////////////////////////////////////////
CRS232Layer::CRS232Layer(void)
: m_hDeviceEnumThread(NULL)
, m_hDeviceEnumThreadReady(NULL)
, m_hwndDeviceEnum(NULL)
, m_pfnNotifyCallback(NULL)
, m_hControlInterfaceReady(NULL)
, m_hControlInterfaceStop(NULL)
, m_hControlInterfaceNotifyEvent(NULL)
, m_hControlInterfaceThread(NULL)

{
  // Initialize critical section for the interface map
  InitializeCriticalSection(&m_tcsInterfaceLock);

  // Clear map
  m_cmInterfaceControlMap.clear();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Destructor of CRS232Layer
///////////////////////////////////////////////////////////////////////////////////////////
CRS232Layer::~CRS232Layer(void)
{
  //Deinit();

  // Destroy critical section for the interface map
  DeleteCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Serach the interface for a given device name
///  \param  szDeviceName   Name of the Interface
///  \return pcIntf on success
///////////////////////////////////////////////////////////////////////////////////////////
CRS232Interface* CRS232Layer::GetInterface(std::string szDeviceName)
{
  CRS232Interface* pcIntf = NULL;

  EnterCriticalSection(&m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iterIF = m_cmInterfaceControlMap.find(szDeviceName);

  if(iterIF != m_cmInterfaceControlMap.end())
  {
    pcIntf = iterIF->second;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return pcIntf;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create a interface with the given name
///  \param  szDeviceName   Name of the Interface
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Layer::CreateInterface(std::string szDeviceName)
{
  /* Get interface configuration */
  g_cRS232Config.Lock();

  CRS232Config::iterator iterConfig = g_cRS232Config.find( szDeviceName.c_str());

  /* If there is no configuration for our interface -> create new one */
  if (g_cRS232Config.end() == iterConfig)
  {
    CString csRegistryKey;
    iterConfig = g_cRS232Config.insert( szDeviceName.c_str());
    g_cRS232Config.StoreToRegistry(csRegistryKey);

    TRACE("%s: IF-Create - interface unknown, create a new one", szDeviceName.c_str());

  }

  DWORD dwBaudRate         = iterConfig.GetBaudRate();
  BYTE  bByteSize          = iterConfig.GetByteSize();
  BYTE  bParity            = iterConfig.GetParity();
  BYTE  bStopBits          = iterConfig.GetStopBits();
  DWORD dwSendTimeout      = iterConfig.GetSendTimeout();
  DWORD dwResetTimeout     = iterConfig.GetResetTimeout();
  DWORD dwKeepAliveTimeout = iterConfig.GetKeepAliveTimeout();
  DWORD dwState            = iterConfig.IsExcluded() ? eINTERFACE_STATE_DISABLED : eINTERFACE_STATE_AVAILABLE;

  TRACE("%s: IF-Create - created with, State: 0x%08x", szDeviceName.c_str(), dwState);

  g_cRS232Config.Unlock();

  // Lock interface map
  EnterCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iter =   g_pcRS232Layer->m_cmInterfaceControlMap.find(szDeviceName);

  if(iter != g_pcRS232Layer->m_cmInterfaceControlMap.end())
  {
    // Interface is available, set state
    // Interface is not more available, set state
    CRS232Interface* pcInterface = iter->second;

    // NOTE: If there are more than one interface for the same device, the same device
    //       will be notified twice (see ::HandleWindowProc()). If the upper layer
    //       connects with the first notication the second notification will overwrite
    //       the interface state from 'running' to 'available', thus it is not possible
    //       to send data anymore.
    // So update interface state only once and ignore the following
    if(pcInterface->UpdateInterfaceState())
    {
      /* disable update of the interface state */
      pcInterface->UpdateInterfaceState(FALSE);
      pcInterface->SetCtrlState(dwState);
    }
    TRACE("%s: IF-Create - multiple interfaces, update disable, State: 0x%08x", szDeviceName.c_str(), dwState);

  } else
  {
    // Interface was not found in control map so create a new entry
    INTERFACE_CONTROL_T tIntContr = {0};

    tIntContr.ulState              = dwState;
    tIntContr.dwBaudRate           = dwBaudRate;
    tIntContr.bByteSize            = bByteSize;
    tIntContr.bParity              = bParity;
    tIntContr.bStopBits            = bStopBits;
    tIntContr.dwSendTimeout        = dwSendTimeout;
    tIntContr.dwResetTimeout       = dwResetTimeout;
    tIntContr.dwKeepAliveTimeout   = dwKeepAliveTimeout;

    CRS232Interface* pcIF = new CRS232Interface( szDeviceName.c_str(), &tIntContr);

    // Insert new entry into interface control map
    g_pcRS232Layer->m_cmInterfaceControlMap.insert(make_pair(szDeviceName, pcIF));

    TRACE("%s: IF-Create - successfully: 0x%08x", szDeviceName.c_str(), pcIF);
  }
  // Unlock interface map
  LeaveCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Layer initialization (Create enumeration thread and load cfgmgr32.dll)
///  \param  pfnDevNotifyCallback   Callback for notifications
///  \param  pvUser                 User pointer
///  \return NXCOM_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Layer::Init( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser)
{
  /* Check if connector is enabled */
  if ( !g_cRS232Config.IsConnectorEnabled())
  {
    TRACE("Connector Init - failed: lRet: NXCON_DRV_DISABLED");
    return NXCON_DRV_DISABLED;
  }

  if( !pfnDevNotifyCallback)
    return NXCON_DRV_INVALID_PARAMETER;

  long lRet = NXCON_NO_ERROR;

  m_pvNotifyUser      = pvUser;
  m_pfnNotifyCallback = pfnDevNotifyCallback;

  // Create  Thread and Events (Enumerate devices)
  m_hDeviceEnumThreadReady  = ::CreateEvent(NULL, FALSE, FALSE, NULL);
  m_hDeviceEnumThread       = ::CreateThread(NULL, 0, DeviceEnumThread, this, 0, NULL);

  if (NULL == m_hDeviceEnumThread)
  {
    TRACE("Connector Init - failed, unable to create device thread, lRet: NXCON_DRV_INIT_ERROR");
    lRet = NXCON_DRV_INIT_ERROR;

  } else
  {
    // Check TransportThread is running
    DWORD dwWaitRes = ::WaitForSingleObject(m_hDeviceEnumThreadReady , INFINITE);

    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:
        lRet = NXCON_NO_ERROR;
      break;

      case WAIT_TIMEOUT:
      {
        lRet = NXCON_DRV_INIT_ERROR;
      }
      break;

      default:
        _ASSERT(false);
      break;
    }

    // Create  Thread and Events (Control Interfaces)
    m_hControlInterfaceReady        = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    m_hControlInterfaceStop         = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    m_hControlInterfaceNotifyEvent  = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    m_hControlInterfaceThread       = ::CreateThread(NULL, 0, ControlInterfaceThread, this, 0, NULL);

    // Check ControlInterfaceThread is running
    dwWaitRes = ::WaitForSingleObject(m_hControlInterfaceReady, CONNECTOR_THREAD_TIMEOUT);

    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:
        lRet = NXCON_NO_ERROR;
      break;

      case WAIT_TIMEOUT:
      {
        TRACE("Connector Init - failed, ControlInterfaceThread not running, lRet: NXCON_DRV_INIT_ERROR");
        lRet = NXCON_DRV_INIT_ERROR;
      }
      break;

      default:
        _ASSERT(false);
      break;
    }
  }


  if(lRet != NXCON_NO_ERROR)
  {
    Deinit();
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Layer deinitialization (Stop device enumeration thread and unload cfgmgr32.dll)
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Layer::Deinit(void)
{
  TRACE("Connector Deinit - Stop device enumeration and unload cfgmgr32.dll");
  DisableDeviceNotification();

  if(NULL != m_hDeviceEnumThread)
  {
    if( NULL != m_hwndDeviceEnum)
    {
      PostMessage( m_hwndDeviceEnum, WM_QUIT, 0, NULL);
    }

    if(WaitForSingleObject(m_hDeviceEnumThread, END_THREAD_WAIT_TIMEOUT) != WAIT_OBJECT_0)
      ::TerminateThread(m_hDeviceEnumThread, MAXDWORD);

    CloseHandle(m_hDeviceEnumThreadReady);

    m_hDeviceEnumThreadReady  = NULL;
    m_hDeviceEnumThread       = NULL;
    m_hwndDeviceEnum          = NULL;
  }

  // Close Interface control mechanism
  if(NULL != m_hControlInterfaceThread)
  {
    // Lock interface map
    EnterCriticalSection(&m_tcsInterfaceLock);

    // Close interface control thread
    SetEvent(m_hControlInterfaceStop);

    // Wait until thread is stopped
    if(WaitForSingleObject(m_hControlInterfaceThread, CONNECTOR_THREAD_TIMEOUT + 10) != WAIT_OBJECT_0)
      ::TerminateThread(m_hControlInterfaceThread, MAXWORD);

    CloseHandle(m_hControlInterfaceReady);
    CloseHandle(m_hControlInterfaceStop);
    CloseHandle(m_hControlInterfaceNotifyEvent);

    m_hControlInterfaceReady        = NULL;
    m_hControlInterfaceStop         = NULL;
    m_hControlInterfaceNotifyEvent  = NULL;
    m_hControlInterfaceThread       = NULL;
  }

  /* Delete interface objects */
  EnterCriticalSection(&m_tcsInterfaceLock);

  for(INTERFACE_MAP::iterator iterIntContr = m_cmInterfaceControlMap.begin(); iterIntContr != m_cmInterfaceControlMap.end(); ++iterIntContr)
  {
    CRS232Interface* pcIntf = iterIntContr->second;

    EnterCriticalSection(&pcIntf->m_tcsInterfaceLock);
    /* We have to send a eDELETED notification to the netXTransport Layer */
    TRACE("%s: LY-Deinit - send delete Notification: eDELETED!", pcIntf->m_szDeviceName.c_str());

    // Signal interface deleted
    m_pfnNotifyCallback( pcIntf->m_szDeviceName.c_str(), eDELETED, this, this->m_pvNotifyUser);
    LeaveCriticalSection(&pcIntf->m_tcsInterfaceLock);

    iterIntContr->second->Stop();
    delete iterIntContr->second;
    iterIntContr->second = NULL;
  }

  m_cmInterfaceControlMap.clear();

  // Unlock interface map
  LeaveCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Thread "Enumerate devices", handle messages loop
///  \param  pvParam            Thread parameters ( CRS232Layer*)
///  \return DWORD  0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CRS232Layer::ControlInterfaceThread(void* pvParam)
{
  CRS232Layer* pcLayer        = reinterpret_cast<CRS232Layer*>(pvParam);
  bool         fThreadRunning = true;

  if(NULL != pcLayer->m_hControlInterfaceNotifyEvent)
  {
    HANDLE  ahWaitEvents[] = { pcLayer->m_hControlInterfaceNotifyEvent,
                               pcLayer->m_hControlInterfaceStop};

    // Create interface notification control map
    INTERFACE_MAP cmIntNotifyMap;
    cmIntNotifyMap.clear();

    INTERFACE_MAP::iterator iterNotify;
    INTERFACE_MAP::iterator iterInterface;

    // Signal Thread ready
    SetEvent(pcLayer->m_hControlInterfaceReady);

    while(fThreadRunning)
    {
      // Wait Inifite -> when thread should close the close event will be signaled
      DWORD dwWaitRes = ::WaitForMultipleObjects( sizeof(ahWaitEvents) / sizeof(ahWaitEvents[0]),
                                                  ahWaitEvents,
                                                  FALSE,
                                                  INFINITE);
      switch(dwWaitRes)
      {
        case WAIT_OBJECT_0: // Receive notification event
        {
          // Lock interface control map
          EnterCriticalSection(&pcLayer->m_tcsInterfaceLock);

          // Iterate over control interface map
          iterInterface = pcLayer->m_cmInterfaceControlMap.begin();

          // iterate over interface control map
          while(iterInterface != pcLayer->m_cmInterfaceControlMap.end())
          {
            CRS232Interface* pcIF = iterInterface->second;

            if( (eINTERFACE_STATE_AVAILABLE == pcIF->GetCtrlState()) ||
                (eINTERFACE_STATE_NOT_AVAIL == pcIF->GetCtrlState()) )
            {
              cmIntNotifyMap.insert(make_pair(iterInterface->first, pcIF));
            }
            iterInterface++;
          }
          LeaveCriticalSection(&pcLayer->m_tcsInterfaceLock);

          // Handle notification map
          while(!cmIntNotifyMap.empty())
          {
            iterNotify = cmIntNotifyMap.begin();

            switch(iterNotify->second->GetCtrlState())
            {
              case eINTERFACE_STATE_NOT_SET:
              case eINTERFACE_STATE_NOT_INIT:
              case eINTERFACE_STATE_RUNNING:
              case eINTERFACE_STATE_STOPPED:
              {
                _ASSERT(false);
              }
              break;

              case eINTERFACE_STATE_AVAILABLE:
              {
                /* Open interface configuration */
                g_cRS232Config.Lock();
                CRS232Config::iterator iterConfig = g_cRS232Config.find( iterNotify->first.c_str());

                /* If there is no configuration for our interface create new one */
                if (g_cRS232Config.end() == iterConfig)
                {
                  CString csRegistryKey;
                  iterConfig = g_cRS232Config.insert( iterNotify->first.c_str());
                  g_cRS232Config.StoreToRegistry(csRegistryKey);
                }
                BOOL fExcluded = iterConfig.IsExcluded();
                g_cRS232Config.Unlock();

                /* Don't signal device notification if interface is black listed */
                if ( !fExcluded &&
                     (pcLayer->m_pfnNotifyCallback != NULL))
                {

                  TRACE("%s: Received Event Interface Notification (ATTACHED)!", iterNotify->first.c_str());

                  // Signal interface is plugged
                  pcLayer->m_pfnNotifyCallback(iterNotify->first.c_str(),
                                               eATTACHED,
                                               pcLayer,
                                               pcLayer->m_pvNotifyUser);
                }
                cmIntNotifyMap.erase(iterNotify);
              }
              break;

              case eINTERFACE_STATE_NOT_AVAIL:
              {
                if(pcLayer->m_pfnNotifyCallback != NULL)
                {
                  CRS232Interface* pcIntf = iterNotify->second;

                  // Stop interface, cause the interface should be closed at first after detection
                  TRACE("%s: Received Event Interface Notification (DETACHED)!", iterNotify->first.c_str());

                  pcIntf->Stop();

                  // Signal interface is unplugged
                  pcLayer->m_pfnNotifyCallback(iterNotify->first.c_str(),
                                               eDETACHED,
                                               pcLayer,
                                               pcLayer->m_pvNotifyUser);
                }
                cmIntNotifyMap.erase(iterNotify);
              }
              break;
            }
          }
        }
        break;

        case WAIT_OBJECT_0 + 1:
        {
          // Received stop event
          fThreadRunning = false;
        }
        break;

        default:
          _ASSERT(false);
        break;
      }
    }

    // Clear interface control map
    cmIntNotifyMap.clear();
  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Thread "Control Interfaces", handle
///  \param  pvParam            Thread parameters ( CRS232Layer*)
///  \return DWORD  0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CRS232Layer::DeviceEnumThread(void* pvParam)
{
  CRS232Layer* pcLayer = reinterpret_cast<CRS232Layer*>(pvParam);

  long            lRet                = NXCON_NO_ERROR;

  if(NXCON_NO_ERROR == (lRet = pcLayer->EnableDeviceNotification()))
  {
    // Enumerate available devices
    pcLayer->EnumerateDevices();

    // Signal thread is running
    SetEvent(pcLayer->m_hDeviceEnumThreadReady);

    // Main message loop:
    MSG msg;
    while (GetMessage(&msg, pcLayer->m_hwndDeviceEnum, 0, 0))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }

  } else
  {
    // Signal thread is running
    SetEvent(pcLayer->m_hDeviceEnumThreadReady);
  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enable all device change notifications
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Layer::EnableDeviceNotification( void)
{
  long lRet = NXCON_DRV_INIT_ERROR;

  if(NULL == m_hwndDeviceEnum)
  {
    // Handle and register pseudo windows
    WNDCLASSEX wcex = {0};

    wcex.cbSize         = sizeof(WNDCLASSEX);
    wcex.style          = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc    = (WNDPROC)HandleWindowProc; // HandleWindowProc;
    wcex.cbClsExtra     = 0;
    wcex.cbWndExtra     = 0;
    wcex.hInstance      = (HINSTANCE)g_hModule;
    wcex.hbrBackground  = NULL;
    wcex.lpszMenuName   = NULL;
    wcex.lpszClassName  = szWindowClass;
    wcex.hIconSm        = NULL;

    /*ATOM Atom           =*/ ::RegisterClassEx(&wcex);

    HWND hWnd = NULL;

    // Create empty window
    hWnd = CreateWindow(szWindowClass,
                        NULL,
                        WS_OVERLAPPEDWINDOW,
                        0,
                        0,
                        0,
                        0,
                        NULL,
                        NULL,
                        wcex.hInstance,
                        NULL);

    m_hwndDeviceEnum = hWnd;

    // Save Window handle
    if(NULL == m_hwndDeviceEnum)
    {
      /* Failed to create window */
      lRet = NXCON_DRV_INIT_ERROR;
    } else
    {
      HDEVNOTIFY                    hDevNotify;
      DEV_BROADCAST_DEVICEINTERFACE NotificationFilter;

      ZeroMemory( &NotificationFilter, sizeof(NotificationFilter) );


      NotificationFilter.dbcc_size       = sizeof(DEV_BROADCAST_DEVICEINTERFACE);
      NotificationFilter.dbcc_devicetype = DBT_DEVTYP_DEVICEINTERFACE;

      /* Register all monitored device interfaces, use GUID*/
      for(int i=0; i<sizeof(GUID_DEVINTERFACE_LIST)/sizeof(GUID); i++)
      {
        NotificationFilter.dbcc_classguid = GUID_DEVINTERFACE_LIST[i];
        hDevNotify = RegisterDeviceNotification(m_hwndDeviceEnum,
                                                &NotificationFilter,
                                                DEVICE_NOTIFY_WINDOW_HANDLE);
        if(!hDevNotify)
        {
          /* Couldn't register device notification */
          DisableDeviceNotification();
          lRet = NXCON_DRV_INIT_ERROR;
        }
        else
        {
          /* Save all register notifications, for each registered GUID */
          m_cvDeviceNotifications.push_back(hDevNotify);
          lRet = NXCON_NO_ERROR;
        }
      }
      ShowWindow(m_hwndDeviceEnum, SW_HIDE);
      UpdateWindow(m_hwndDeviceEnum);
    }
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Disable all device change notifications
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Layer::DisableDeviceNotification( void)
{
  /* Remove all registered notifications */
  while (!m_cvDeviceNotifications.empty())
  {
    HDEVNOTIFY hDevNotify = m_cvDeviceNotifications.back();

    if (NULL != hDevNotify)
    {
      /* Unregister device notification (notification for each GUID)  */
      UnregisterDeviceNotification(hDevNotify);
      m_cvDeviceNotifications.pop_back();
    }
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enumeration window message handler
///  \param hwnd
///  \param uMsg
///  \param wParam
///  \param lParam
///  \return
///////////////////////////////////////////////////////////////////////////////////////////
LRESULT CALLBACK CRS232Layer::HandleWindowProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  LRESULT lRet = TRUE;

  switch (uMsg)
  {
    case WM_DEVICECHANGE:
    {
      /* Check for device changed */
      switch(wParam)
      {
        case DBT_DEVICEARRIVAL:
        {
          PDEV_BROADCAST_HDR pHdr = (PDEV_BROADCAST_HDR)lParam;

          /* Check if we have a new Port (serial device) */
          if ( DBT_DEVTYP_DEVICEINTERFACE == pHdr->dbch_devicetype)
          {
            TRACE("**** DBT_DEVICEARRIVAL (Type: %d) ****", pHdr->dbch_devicetype);

            PDEV_BROADCAST_DEVICEINTERFACE pDevInf = (PDEV_BROADCAST_DEVICEINTERFACE)pHdr;

            std::string szDeviceName;
            if(g_pcRS232Layer->ExtractDeviceName(pDevInf, wParam, szDeviceName))
            {
              g_pcRS232Layer->CreateInterface(szDeviceName);
              TRACE("%s: Set Event Interface Notification plugged", szDeviceName.c_str());

              // Set Event to signal something is happening
              SetEvent(g_pcRS232Layer->m_hControlInterfaceNotifyEvent);
            }
          }
        }
        break;

        case DBT_DEVICEREMOVEPENDING:
        case DBT_DEVICEREMOVECOMPLETE:
        {
          PDEV_BROADCAST_HDR pHdr = (PDEV_BROADCAST_HDR)lParam;

          /* Check if we have a Port (serial device) which disapears */
          if ( DBT_DEVTYP_DEVICEINTERFACE == pHdr->dbch_devicetype)
          {
            TRACE("**** DBT_DEVICEREMOVE (Type: %d) ****", pHdr->dbch_devicetype);

            PDEV_BROADCAST_DEVICEINTERFACE pDevInf = (PDEV_BROADCAST_DEVICEINTERFACE)pHdr;

            std::string szDeviceName;
            if(g_pcRS232Layer->ExtractDeviceName(pDevInf, wParam, szDeviceName))
            {
              // Lock interface map
              EnterCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);

              INTERFACE_MAP::iterator iter = g_pcRS232Layer->m_cmInterfaceControlMap.find(szDeviceName);
              if(iter != g_pcRS232Layer->m_cmInterfaceControlMap.end())
              {
                // Interface is not more available, set state
                CRS232Interface* pcInterface = iter->second;

                pcInterface->SetCtrlState(eINTERFACE_STATE_NOT_AVAIL);

                // NOTE: If there are more than one interface for the same device, the same device
                //       will be notified multiple times. If the upper layer connects with the first notification
                //       the second notification will overwrite the interface state from 'running' to
                //       'available' (see ::CreateInterface()), thus it is not possible to send data anymore.
                // Enable update of the interface state for next attach event; the interface update will be disabled
                // after the first interface creation is done (see ::CreateInterface()).
                pcInterface->UpdateInterfaceState(TRUE);

                TRACE("%s: Set Event Interface Notification unplugged", szDeviceName.c_str());

                // Set Event to signal something is happening
                SetEvent(g_pcRS232Layer->m_hControlInterfaceNotifyEvent);
              }

              // Unlock interface map
              LeaveCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);
            }
          }
        }
        break;

        default:
          lRet = DefWindowProc(hwnd, uMsg, wParam, lParam);
        break;
      } /* end switch */
    }
    break;

    case WM_DESTROY:
      PostQuitMessage(0);
    break;

    case WM_QUIT:
      DestroyWindow( hwnd);
    break;

    default:
      // Send all other messages on to the default windows handler.
      lRet = DefWindowProc(hwnd, uMsg, wParam, lParam);
    break;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Extract device name from device interface via registry
///  \param pDevInf
///  \param wParam
///  \param szDeviceName  Returned device name
///  \return true on success
///////////////////////////////////////////////////////////////////////////////////////////
bool CRS232Layer::ExtractDeviceName(PDEV_BROADCAST_DEVICEINTERFACE  pDevInf,
                                    WPARAM                          wParam,
                                    std::string&                    szDeviceName)
{
  // EXAMPLE:
  // pDevInf->dbcc_name:
  // \\?\USB#Vid_04e8&Pid_503b#0002F9A9828E0F06#{a5dcbf10-6530-11d2-901f-00c04fb951ed}
  // szDevId: USB\Vid_04e8&Pid_503b\0002F9A9828E0F06
  // szClass: USB

  HDEVINFO hDevInfo = NULL;
  bool     fRet     = false;
  bool     fFound   = false;

  SP_DEVINFO_DATA spDevInfoData;
  spDevInfoData.cbSize = sizeof(SP_DEVINFO_DATA);

  if(NULL == pDevInf)
    return false;

  /* Change device name to a readable form */
  CString szDevId = pDevInf->dbcc_name+4;
  int     idx     = szDevId.ReverseFind(_T('#'));

  /*  Check number of idx (#) */
  if( -1 != idx)
  {
    szDevId.Truncate(idx);
    szDevId.Replace(_T('#'), _T('\\'));
    szDevId.MakeUpper();

    CString szClass;
    idx = szDevId.Find(_T('\\'));
    // Check number of idx (\\)
    szClass = szDevId.Left(idx);
    // Seems we should ignore "ROOT" type....
    if ( _T("ROOT") != szClass )
    {
      // Depends on the wParam we don't need all classes ;-)
      DWORD dwFlag = DBT_DEVICEARRIVAL != wParam ? DIGCF_ALLCLASSES : (DIGCF_ALLCLASSES | DIGCF_PRESENT);

      // Get and check handle to the device class
      if(NULL != (hDevInfo = SetupDiGetClassDevs(NULL,szClass,NULL,dwFlag)))
      {
        // Search device in device class folder
        // Iterate, over all devices until our searched devices was found
        for(int i=0; SetupDiEnumDeviceInfo(hDevInfo, i, &spDevInfoData); i++)
        {
          DWORD nSize=0 ;
          TCHAR buf[MAX_PATH];

          // Get device id
          if( !SetupDiGetDeviceInstanceId(hDevInfo, &spDevInfoData, buf, sizeof(buf), &nSize) )
            break;
          // Check, that the found device is our device
          if ( szDevId == buf )
          {
            // OK, device found
            #if _DEBUG
              OLECHAR* bstrGuid;
              StringFromCLSID(spDevInfoData.ClassGuid, &bstrGuid);
              TRACE("Device Class GUID: %S", bstrGuid);
              ::CoTaskMemFree(bstrGuid);
              StringFromCLSID(pDevInf->dbcc_classguid, &bstrGuid);
              TRACE("Device Interface GUID: %S", bstrGuid);
              ::CoTaskMemFree(bstrGuid);
              TRACE("Device ID: %s", szDevId.GetBuffer());
              TRACE("Device Class: %s", szClass.GetBuffer());
            #endif
            fFound = true;
            break;
          }
        }
      }
    }
  }

  if(fFound)
  {
    if (spDevInfoData.ClassGuid == GUID_DEVCLASS_PORTS)
    {
      HKEY hkDevice = SetupDiOpenDevRegKey(hDevInfo, &spDevInfoData, DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_QUERY_VALUE);

      if (!hkDevice)
      {
        TRACE("SetupDiOpenDevRegKey() failed - LastError = 0x%08x", GetLastError());

      } else
      {

        TCHAR szName[MAX_PATH];
        DWORD dwLen   = sizeof(szName);
        LONG  lRegRet = ERROR_SUCCESS;

        // Init the string
        szDeviceName.assign("UNKNOWN");

        // Get value from the registry
        if( ERROR_SUCCESS == (lRegRet = ::RegQueryValueEx( hkDevice, _T("Portname"),NULL,
                                                           NULL,(BYTE*)szName, &dwLen)) )
        {
          // Set device name
          szDeviceName = szName;
          fRet = true;
        }

        TRACE("Interface name: %s, (RegQueryValueEx(), Error = 0x%08x)", szDeviceName.c_str(), lRegRet);

        // Close Reg Key
        ::RegCloseKey(hkDevice);
      }
    }
  }

  return fRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enumerate for COM devices
///  \return 0 on success
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Layer::EnumerateDevices( void)
{
  TRACE("Enumerate Devices - initial serial port enumeration");

  long          lRet = ERROR_SUCCESS;
  CRegKey       cRegBaseKey;
  CString       csRegPath;
  csRegPath.Format( _T("HARDWARE\\DEVICEMAP\\SERIALCOMM"));

  if ( ERROR_SUCCESS != (lRet = cRegBaseKey.Open(HKEY_LOCAL_MACHINE, csRegPath, KEY_READ )))
  {
    DWORD dwLastError = GetLastError();
    TRACE("Enumerate Devices - failed to open registry-key: %s, lRet = 0x%08x, LastError: 0x%08x", (LPCTSTR)csRegPath.GetBuffer(), lRet, dwLastError);
  }else
  {
    DWORD   dwI = 0;
    TCHAR   achValueName[MAX_PATH];

    while (lRet != ERROR_NO_MORE_ITEMS)
    {
      DWORD dwValue = MAX_PATH;
      if( ERROR_SUCCESS == (lRet = RegEnumValue(cRegBaseKey.m_hKey, dwI, achValueName, &dwValue, NULL, NULL, NULL, NULL)))
      {
        /* We found an entry, now we have to read the value */
        TCHAR   szPortName[MAX_PATH];
        ULONG   nChars = MAX_PATH;
        if( ERROR_SUCCESS == cRegBaseKey.QueryStringValue( achValueName, szPortName, &nChars))
        {
          TRACE("**** Found: %s ****", szPortName);
          CreateInterface((LPCTSTR)szPortName);

          if(m_pfnNotifyCallback)
            m_pfnNotifyCallback(szPortName, eATTACHED, this, m_pvNotifyUser);
        }
      }
      dwI++;
    }

    /* Close registry base key */
    cRegBaseKey.Close();
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
///  Validate the given interface pointer
///   \param  pvInterface  Pointer to interface object
///   \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Layer::ValidateInterface (PCONNECTOR_INTERFACE pvInterface)
{
  long lRet = NXCON_DRV_INVALID_POINTER;

  // Lock interface map
  EnterCriticalSection(&m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iter = m_cmInterfaceControlMap.begin();

  while(iter != m_cmInterfaceControlMap.end())
  {
    if(iter->second == reinterpret_cast<CRS232Interface*>(pvInterface))
    {
      lRet = NXCON_NO_ERROR;
      break;
    }
    iter++;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
///  Re-initialize RS232 connector
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Layer::Reinit(void)
{
  //Lock interface control map
  EnterCriticalSection(&m_tcsInterfaceLock);

  // Iterate over control interface map
  INTERFACE_MAP::iterator iterInterface = m_cmInterfaceControlMap.begin();

  // iterate over interface control map
  while(iterInterface != m_cmInterfaceControlMap.end())
  {
    unsigned long ulState  = iterInterface->second->GetCtrlState();
    CRS232Config::iterator iterConfig = g_cRS232Config.find( iterInterface->first.c_str());

    if(iterConfig != g_cRS232Config.end())
    {
      // Stop Interface
      CRS232Interface* pcIntf = iterInterface->second;

      if(iterConfig.IsExcluded())
      {
        // Signal interface is unplugged
        m_pfnNotifyCallback( iterInterface->first.c_str(), eDETACHED, this, m_pvNotifyUser);

        pcIntf->Stop();

        /* Disallow starting of this interface */
        pcIntf->SetCtrlState(eINTERFACE_STATE_DISABLED);

      } else if(ulState == eINTERFACE_STATE_DISABLED)
      {
        /* Allow starting of this interface */
        pcIntf->SetCtrlState(eINTERFACE_STATE_AVAILABLE);
      }

      if( (pcIntf->GetBaudRate()         != iterConfig.GetBaudRate())     ||
          (pcIntf->GetByteSize()         != iterConfig.GetByteSize())     ||
          (pcIntf->GetParity()           != iterConfig.GetParity())       ||
          (pcIntf->GetStopBits()         != iterConfig.GetStopBits())     ||
          (pcIntf->GetSendTimeout()      != iterConfig.GetSendTimeout())  ||
          (pcIntf->GetResetTimeout()     != iterConfig.GetResetTimeout()) ||
          (pcIntf->GetKeepAliveTimeout() != iterConfig.GetKeepAliveTimeout())  )
      {
        /* Reconfigure physical connection */
        pcIntf->SetBaudRate( iterConfig.GetBaudRate());
        pcIntf->SetByteSize( iterConfig.GetByteSize());
        pcIntf->SetParity(   iterConfig.GetParity());
        pcIntf->SetStopBits( iterConfig.GetStopBits());
        pcIntf->SetSendTimeout( iterConfig.GetSendTimeout());
        pcIntf->SetResetTimeout( iterConfig.GetResetTimeout());
        pcIntf->SetKeepAliveTimeout( iterConfig.GetKeepAliveTimeout());

        if(ulState == eINTERFACE_STATE_RUNNING)
        {
          PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveCallback = pcIntf->m_pfnReceiveCallback;
          void*                               pvReceiveUserData  = pcIntf->m_pvReceiveUserData;

          pcIntf->Stop();
          pcIntf->Start(pfnReceiveCallback, pvReceiveUserData);
        }
      }
    }
    iterInterface++;
  }
  LeaveCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
///   CRS232Interface
///////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor
///   \param szIntfName             Name of the interface
///   \param ptInterfaceControlData Interface control structure
///////////////////////////////////////////////////////////////////////////////////////////
CRS232Interface::CRS232Interface(const char* szIntfName, INTERFACE_CONTROL_T* ptInterfaceControlData)
: m_hComPort(INVALID_HANDLE_VALUE)
, m_hComThread(NULL)
, m_hComThreadStop(NULL)
, m_hComThreadReady(NULL)
, m_fNotify(TRUE)
{
  InitializeCriticalSection(&m_tcsInterfaceLock);

  m_tInterfaceControl = *ptInterfaceControlData;
  m_szDeviceName      = szIntfName;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Destructor
///////////////////////////////////////////////////////////////////////////////////////////
CRS232Interface::~CRS232Interface()
{
  DeleteCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Start interface (Open COM port and start all needed threads)
///   \param  pfnReceiveCallback    Callback for the receiving data
///   \param  pvUser                User pointer
///   \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Interface::Start(PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveCallback, void* pvUser)
{
  long lRet = NXCON_NO_ERROR;

  /* Check if interface is disabled */
  if(eINTERFACE_STATE_DISABLED == m_tInterfaceControl.ulState)
  {
    TRACE("%s: IF-Start - ulState: eINTERFACE_STATE_DISABLED lRet: 0x%08x", m_szDeviceName.c_str(), NXCON_DRV_DISABLED);
    return NXCON_DRV_DISABLED;
  }

  if( ( (eINTERFACE_STATE_STOPPED    == m_tInterfaceControl.ulState)  ||
        (eINTERFACE_STATE_NOT_INIT   == m_tInterfaceControl.ulState)  ||
        (eINTERFACE_STATE_AVAILABLE  == m_tInterfaceControl.ulState)) )
  {
    _ASSERT(INVALID_HANDLE_VALUE == m_hComPort);

    m_pvReceiveUserData   = pvUser;
    m_pfnReceiveCallback  = pfnReceiveCallback;
    m_hComThreadStop      = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    m_hComThreadReady     = ::CreateEvent(NULL, FALSE, FALSE, NULL);

    if(NXCON_NO_ERROR == (lRet = OpenPort()))
    {
      if(NULL == (m_hComThread = ::CreateThread(NULL, 0, CommThread, this, 0, NULL)))
      {
        /* Error creating handler thread */
        lRet = NXCON_DRV_NOT_START;
      }else
      {
        DWORD dwWaitRes = ::WaitForSingleObject(m_hComThreadReady, CONNECTOR_THREAD_TIMEOUT);

        switch(dwWaitRes)
        {
          case WAIT_OBJECT_0:
          {
            m_tInterfaceControl.ulState = eINTERFACE_STATE_RUNNING;
          }
          break;

          case WAIT_TIMEOUT:
          {
            m_tInterfaceControl.ulState = eINTERFACE_STATE_STOPPED;
          }
          break;

          default:
          _ASSERT(false);
          break;
        }
      }
    } else
    {
      // Change state, cause it was possible to open or config port
      m_tInterfaceControl.ulState = eINTERFACE_STATE_STOPPED;
    }

    if(NXCON_NO_ERROR != lRet)
    {
      TRACE("%s: IF-Start - Failed, ulState: %d lRet: 0x%08x", m_szDeviceName.c_str(), m_tInterfaceControl.ulState, lRet);
      Stop();
    }else
    {
      TRACE("%s: IF-Start - Started, ulState: %d lRet: 0x%08x", m_szDeviceName.c_str(), m_tInterfaceControl.ulState, lRet);
    }
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Stop interface (Close COM port and stop all threads)
///////////////////////////////////////////////////////////////////////////////////////////
void CRS232Interface::Stop( void)
{
  /* Check if interface is disabled */
  if(eINTERFACE_STATE_DISABLED == m_tInterfaceControl.ulState)
  {
    TRACE("%s: IF-Stop - ulState: eINTERFACE_STATE_DISABLED", m_szDeviceName.c_str());
    return;
  }

  /* Check if interface is already stopped */
  if(eINTERFACE_STATE_STOPPED == m_tInterfaceControl.ulState)
  {
    TRACE("%s: IF-Stop - ulState: eINTERFACE_STATE_STOPPED", m_szDeviceName.c_str());
    return;
  }

  m_tInterfaceControl.ulState = eINTERFACE_STATE_STOPPED;
  TRACE("%s: IF-Stop - ulState: eINTERFACE_STATE_STOPPED", m_szDeviceName.c_str());

  /* Stop Interface */
  EnterCriticalSection(&m_tcsInterfaceLock);

  /* Kill any pending receiver thread */
  if(NULL != m_hComThread)
  {
    SetEvent(m_hComThreadStop);

    if(WaitForSingleObject(m_hComThread, END_THREAD_WAIT_TIMEOUT) != WAIT_OBJECT_0)
      ::TerminateThread(m_hComThread, MAXDWORD);

    CloseHandle(m_hComThreadStop);
    m_hComThreadStop = NULL;

    CloseHandle(m_hComThread);
    m_hComThread = NULL;

    CloseHandle(m_hComThreadReady);
    m_hComThreadReady = NULL;
  }

  /* Close open COM port */
  if(INVALID_HANDLE_VALUE != m_hComPort)
  {
    ::CloseHandle(m_hComPort);
    m_hComPort = INVALID_HANDLE_VALUE;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Send Data to interface
///   \param pabData    Data to send
///   \param ulDataLen  Length of data to send
///   \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Interface::Send(unsigned char* pabData, unsigned long ulDataLen)
{
  long lRet = NXCON_NO_ERROR;

  if( eINTERFACE_STATE_RUNNING == m_tInterfaceControl.ulState)
  {
    HANDLE        hOvWriteEvent  = ::CreateEvent(NULL, TRUE, FALSE, NULL);
    OVERLAPPED    tOvWrite       = {0};
    unsigned long ulWritten      = 0;

    lRet = NXCON_DRV_SEND_ERROR;

    tOvWrite.hEvent = hOvWriteEvent;

    if(::WriteFile(m_hComPort, pabData, ulDataLen, &ulWritten, &tOvWrite))
    {
      lRet = NXCON_NO_ERROR;
    } else if(ERROR_IO_PENDING == GetLastError())
    {
      if(WaitForSingleObject(tOvWrite.hEvent, m_tInterfaceControl.dwSendTimeout) != WAIT_OBJECT_0)
      {
        /* Error sending packet */
        ::CancelIo(m_hComPort);
        lRet = NXCON_DRV_SEND_ERROR;
        TRACE("%s: Send - lRet: 0x%08x, LastError:0x%08x", m_szDeviceName.c_str(), lRet, GetLastError());

      } else
      {
        lRet = NXCON_NO_ERROR;
      }
    } else
    {
      ::CancelIo(m_hComPort);
      lRet = NXCON_DRV_SEND_ERROR;
      TRACE("%s: Send - lRet: 0x%08x, LastError:0x%08x", m_szDeviceName.c_str(), lRet, GetLastError());
    }

    ::CloseHandle(hOvWriteEvent);

  }else
  {
    lRet = NXCON_DRV_NOT_INITIALIZED;
    TRACE("%s: Send - lRet: 0x%08x, State: %d", m_szDeviceName.c_str(), lRet, m_tInterfaceControl.ulState);
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get name of the used interface ( COM3/ COM 19  etc.)
///  \param ulSize       Size of the string buffer
///  \param szInterface  Reference for the result string
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Interface::GetInterfaceName( unsigned long ulSize, char* szInterface)
{
  long lRet = NXCON_NO_ERROR;
  if(NULL == szInterface)
  {
    lRet = NXCON_DRV_INVALID_PARAMETER;
  } else if (m_szDeviceName.length() >= ulSize)
  {
    lRet = NXCON_DRV_BUFFER_TOO_SHORT;
  } else
  {
    strncpy(szInterface, m_szDeviceName.c_str(), ulSize);
  }
  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Communication thread wrapper (Receiver)
///  \param pvParam Pointer to object instance
///  \return 0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CRS232Interface::CommThread(void* pvParam)
{
  CRS232Interface* pcInterface = reinterpret_cast<CRS232Interface*>(pvParam);

  return pcInterface->CommThreadFunc();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Communication thread (Receiver)
///  \return 0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CRS232Interface::CommThreadFunc(void)
{
  bool          fRunning      = true;
  OVERLAPPED    tOvWait       = {0};
  HANDLE        hOvWaitEvent  = ::CreateEvent(NULL, TRUE, FALSE, NULL);
  OVERLAPPED    tOvRead       = {0};
  HANDLE        hOvReadEvent  = ::CreateEvent(NULL, TRUE, FALSE, NULL);

  DWORD         dwEvent;

  tOvWait.hEvent  = hOvWaitEvent;
  tOvRead.hEvent  = hOvReadEvent;

  HANDLE ahWaitHandles[] = {hOvWaitEvent, m_hComThreadStop};

  SetEvent(m_hComThreadReady);

  while(fRunning)
  {
    memset(&tOvWait, 0, sizeof(tOvWait));

    bool fWaitSuccess = false;
    tOvWait.hEvent    = hOvWaitEvent;

    if ( ::WaitCommEvent(m_hComPort, &dwEvent, &tOvWait))
    {
      fWaitSuccess = true;
    } else
    {
      DWORD dwErr = 0;
      if( (dwErr = GetLastError()) != ERROR_IO_PENDING)
      {
        TRACE("%s: CommThreadFunc - Wait failed, LastError:0x%08x", m_szDeviceName.c_str(), dwErr);
      } else
      {
        DWORD dwRes = ::WaitForMultipleObjects(sizeof(ahWaitHandles) / sizeof(ahWaitHandles[0]),
                                               ahWaitHandles, FALSE, INFINITE);

        switch(dwRes)
        {
        case WAIT_OBJECT_0:
          {
            dwEvent = 0;
            ::GetCommMask(m_hComPort, &dwEvent);
            ::ResetEvent(tOvWait.hEvent);
            fWaitSuccess = true;
          }
          break;

        case WAIT_OBJECT_0 + 1:
          fRunning = false;
          break;

        case WAIT_TIMEOUT:
          break;

        default:
          _ASSERT(false);
          break;
        }
      }
    }

    if(fWaitSuccess)
    {
      DWORD   dwErrorMask;
      COMSTAT tComstat = {0};

      if(::ClearCommError (m_hComPort, &dwErrorMask, &tComstat) == TRUE)
      {
        if(dwEvent & EV_RXCHAR)
        {
          unsigned long ulReadLen = sizeof(m_abRecvBuffer);
          unsigned long ulReadCnt = 0;

          memset(&tOvRead, 0, sizeof(tOvRead));
          ::ResetEvent(hOvReadEvent);
          tOvRead.hEvent = hOvReadEvent;

          BOOL fRead = ::ReadFile(m_hComPort, m_abRecvBuffer, ulReadLen, &ulReadCnt, &tOvRead);

          if(!fRead)
          {
            if(GetLastError() == ERROR_IO_PENDING)
            {
              if(WAIT_OBJECT_0 == ::WaitForSingleObject(tOvRead.hEvent, 1000))
              {
                BOOL fGetOverlapped = FALSE;
                if ( FALSE == (fGetOverlapped = GetOverlappedResult(m_hComPort, &tOvRead, &ulReadCnt, FALSE)) )
                {
                  TRACE("%s: CommThreadFunc - GetOverlappedResult, Error: 0x%08x", m_szDeviceName.c_str(), GetLastError());
                }else
                {
                  fRead = TRUE;
                }
              }
            } else
            {
              ::CancelIo(m_hComPort);
              TRACE("%s: CommThreadFunc - Receive failed, Error: 0x%08x", m_szDeviceName.c_str(), GetLastError());
            }
          }

          if(fRead && (ulReadCnt > 0))
          {
            // Signal data received
            if( m_pfnReceiveCallback)
            {
              //TRACE("%s: CommThreadFunc - recvCallback, ulReadCnt: 0x%08X, ulReadLen: 0x%08x", m_szDeviceName.c_str(), ulReadCnt, ulReadLen);
              m_pfnReceiveCallback(m_abRecvBuffer, ulReadCnt, m_pvReceiveUserData);
              //TRACE("%s: CommThreadFunc - recvCallback was called \n", m_szDeviceName.c_str());
            }
          }
        }
      }
    }

    // Just check if we have to leave, don't wait
    if(WaitForSingleObject(m_hComThreadStop, 0) == WAIT_OBJECT_0)
      fRunning = false;
  }

  ::CloseHandle(hOvWaitEvent);
  ::CloseHandle(hOvReadEvent);

  PurgeComm(m_hComPort, PURGE_RXCLEAR | PURGE_TXCLEAR);

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Helper function to open and setup the COM port
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Interface::OpenPort( void)
{
  long  lRet    = NXCON_DRV_NOT_OPENED;
  DWORD dwError = 0;

  std::string szPortName = PREFIX_COM_INTERFACE;
  szPortName.append( m_szDeviceName);

  if(INVALID_HANDLE_VALUE != (m_hComPort = ::CreateFile(szPortName.c_str(),
                                                        GENERIC_READ | GENERIC_WRITE,
                                                        0,
                                                        NULL,
                                                        OPEN_EXISTING,
                                                        FILE_FLAG_OVERLAPPED,
                                                        NULL)))
  {
    if(NXCON_NO_ERROR != (lRet = SetupConnection( m_tInterfaceControl.dwBaudRate,
                                                  m_tInterfaceControl.bByteSize,
                                                  m_tInterfaceControl.bParity,
                                                  m_tInterfaceControl.bStopBits)))
    {
      ::CloseHandle(m_hComPort);
      m_hComPort = INVALID_HANDLE_VALUE;
    }
  }

  dwError = GetLastError();
  TRACE("%s: OpenPort - lRet: 0x%08x / LastError: 0x%08x, m_hComPort: 0x%08X", szPortName.c_str(), lRet, dwError, *((unsigned long*)&m_hComPort));

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Setup COM port parameters
///  \param dwBaudRate  CBR_XXX baudrate define
///  \param bByteSize   Bytesize on bus
///  \param bParity     Parity
///  \param bStopBits   Number of stop bits
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CRS232Interface::SetupConnection(DWORD dwBaudRate, BYTE bByteSize, BYTE bParity, BYTE bStopBits)
{
  DCB     dcb                 = {0};
  long    lRet                = NXCON_NO_ERROR;
  COMMTIMEOUTS  tComTimeouts  = {0};

  if(!::GetCommState(m_hComPort, &dcb))
  {
    // handle error
    lRet = NXCON_DRV_GETCOMMSTATE;
  } else
  {
    // Setup COM port data
    dcb.BaudRate          = dwBaudRate;      // real com port = 115200, netSTICK = 256000
    dcb.ByteSize          = bByteSize;       // 8 Bit
    dcb.Parity            = bParity;         // NOPARITY
    dcb.StopBits          = bStopBits;       // ONESTOPBIT

    // other various settings
    dcb.fBinary           = TRUE;
    dcb.fParity           = FALSE;

    // setup hardware flow control
    dcb.fOutxCtsFlow      = FALSE;                  // CTS output flow control
    dcb.fOutxDsrFlow      = FALSE;                  // DSR output flow control
    dcb.fDtrControl       = DTR_CONTROL_DISABLE;    // DTR flow control type
    dcb.fDsrSensitivity   = FALSE;                  // DSR sensitivity
    dcb.fTXContinueOnXoff = FALSE;                  // XOFF continues Tx
    dcb.fOutX             = FALSE;                  // XON/XOFF out flow control
    dcb.fInX              = FALSE;                  // XON/XOFF in flow control
    dcb.fErrorChar        = FALSE;                  // enable error replacement
    dcb.fNull             = FALSE;                  // enable null stripping
    dcb.fRtsControl       = RTS_CONTROL_DISABLE;    // RTS flow control
    dcb.fAbortOnError     = FALSE;                  // ??? abort reads/writes on error

    dcb.XonLim            = 0;
    dcb.XoffLim           = 0;
    dcb.XonChar           = 0;
    dcb.XoffChar          = 0;

    if(!::SetCommState(m_hComPort, &dcb))
    {
      // handle error
      lRet = NXCON_DRV_SETCOMMSTATE;

    } else if(!::GetCommTimeouts(m_hComPort, &tComTimeouts))
    {
      // handle error
      lRet = NXCON_DRV_GETTIMEOUT;
    } else
    {
      // Set COM timeouts
      tComTimeouts.ReadIntervalTimeout          = MAXDWORD;   // This instructs  ReadFile to return immediately
      tComTimeouts.ReadTotalTimeoutMultiplier   = 0;          // with the received charcters.
      tComTimeouts.ReadTotalTimeoutConstant     = 0;          // ReadTotalTimeoutMultiplier and ReadTotalTimeoutConstant
                                                              // must be 0
      tComTimeouts.WriteTotalTimeoutMultiplier  = 0;
      tComTimeouts.WriteTotalTimeoutConstant    = 0;

      // Set the COM timeouts
      if(!::SetCommTimeouts(m_hComPort, &tComTimeouts))
      {
        // handle error
        lRet = NXCON_DRV_SETTIMEOUT;

      // Setup the Queue sizes of input and output buffer
      } else if(!::SetupComm(m_hComPort, 8192, 8192))
      {
        // handle error
        lRet = NXCON_DRV_SETUPCOMBUFFER;

      // Set Event mask for the COM port
      } else if (!::SetCommMask( m_hComPort, (EV_RXCHAR /*| EV_TXEMPTY*/)))   // We do not handle EV_TXEMPTY events
      {
        // handle error
        lRet = NXCON_DRV_SETUPCOMMASK;

      // Signal ready even if DTR and RTS is switched off, a virtual COM port needs this to send the data
      } else  if( !::EscapeCommFunction(m_hComPort, SETDTR) ||
                  !::EscapeCommFunction(m_hComPort, SETRTS) )
      {
        lRet = NXCON_DRV_SETUPCOMHWSIGNAL;
      }
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
